<?php
// public/index.php
declare(strict_types=1);
header('Content-Type: text/html; charset=utf-8');

$root = dirname(__DIR__);
$data_file = $root . '/data/guide.json';
$json = file_exists($data_file) ? file_get_contents($data_file) : '{}';
$data = json_decode($json, true) ?: [];

$site_title = $data['site_title'] ?? 'راهنمای تحویل و پرداخت';
$sections = $data['sections'] ?? [];
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width,initial-scale=1" />
<title><?php echo htmlspecialchars($site_title, ENT_QUOTES, 'UTF-8'); ?></title>
<link rel="icon" type="image/png" href="favicon.png" sizes="512x512">

<style>
@font-face {
  font-family: 'IRANYekan';
  src: url('./font/IRANYekanXRegular.woff2') format('woff2');
  font-display: swap;
}
@font-face {
  font-family: 'IRANYekan';
  src: url('./font/IRANYekanXBold.woff2') format('woff2');
  font-weight: bold;
  font-display: swap;
}

/* Root colors */
:root {
  --bg:#f7f8fb;
  --card:#fff;
  --text:#1f2937;
  --muted:#6b7280;
  --brand-blue:#2563eb;
  --brand-green:#16a34a;
  --brand-yellow:#f59e0b;
  --brand-red:#ef4444;
  --radius:14px;
  --shadow:0 6px 20px rgba(0,0,0,.06);
}
html[data-theme="dark"] {
  --bg:#0f172a;
  --card:#1e293b;
  --text:#f1f5f9;
  --muted:#94a3b8;
  --shadow:0 6px 20px rgba(255,255,255,.06);
}

* { box-sizing: border-box; }
body {
  margin:0;
  font-family:"IRANYekan",system-ui;
  background:var(--bg);
  color:var(--text);
  transition:background .3s,color .3s;
  overflow-x:hidden;
}

/* Header */
.header {
  display:flex;flex-direction:column;align-items:center;gap:10px;
  margin:18px 0;
}
.logo-stripes {
  display:flex;gap:6px;align-items:center;
  background:var(--card);
  padding:6px 12px;border-radius:14px;
  box-shadow:var(--shadow);
}
.logo-stripes span {
  display:block;width:8px;height:20px;border-radius:20px;
  opacity:.85; transition:opacity .2s;
}
.logo-stripes:hover span { opacity:1; }
.b{background:var(--brand-blue)}.g{background:var(--brand-green)}
.y{background:var(--brand-yellow)}.r{background:var(--brand-red)}

/* Theme toggle */
.theme-toggle{
  position:fixed; inset-inline-end:14px; inset-block-start:14px; z-index:120;
  background:var(--card); border:1px solid #e5e7eb; border-radius:10px; padding:6px 10px; cursor:pointer;
}

/* Mega Menu */
nav.toc {
  position:sticky;top:0;z-index:100;
  background:rgba(247,248,251,.96);
  border-bottom:1px solid #e5e7eb;
  backdrop-filter:blur(8px);
  transition:all .3s ease;
}
html[data-theme="dark"] nav.toc {
  background:rgba(15,23,42,.9);
  border-color:#334155;
}
nav.toc.shrink {
  opacity:0.9;
  height:48px;
  box-shadow:0 4px 12px rgba(0,0,0,.12);
}
.toc-inner {
  max-width:960px;margin:auto;padding:10px;
  display:flex;gap:8px;flex-wrap:wrap;justify-content:center;
  flex-direction: row-reverse;
}
.toc a {
  display:inline-block;
  padding:6px 10px;
  border-radius:8px;
  background:var(--card);
  color:var(--text);
  font-size:13px;
  text-decoration:none;
  border:1px solid #e5e7eb;
  transition:.25s;
}
.toc a:hover {background:var(--brand-blue);color:#fff;}
.toc a.active {background:var(--brand-red);color:#fff;font-weight:bold;}

/* Content layout */
.wrap {max-width:960px;margin:auto;padding:16px;}
.card {
  background:var(--card);
  border:1px solid #e5e7eb;
  border-radius:var(--radius);
  box-shadow:var(--shadow);
  margin:14px 0;
  overflow:hidden;
}
details summary {
  list-style:none;cursor:pointer;
  padding:14px;background:#f9fafb;border-bottom:1px solid #e5e7eb;
  font-weight:700;display:flex;justify-content:space-between;align-items:center;
}
html[data-theme="dark"] details summary {background:#1e293b;border-color:#334155;}
details[open] {box-shadow:0 0 0 3px rgba(37,99,235,.25);}
.content {padding:12px 10px;animation:fade .25s ease;}
@keyframes fade {from{opacity:0;transform:translateY(-6px)}to{opacity:1;transform:none}}
p, li {text-align:justify;line-height:1.6;}
ul{padding-right:22px}

/* Back to top */
#toTop {
  position:fixed;bottom:24px;left:24px;
  background:var(--brand-blue);color:#fff;
  border:none;border-radius:50%;
  width:48px;height:48px;display:none;
  align-items:center;justify-content:center;
  cursor:pointer;box-shadow:0 6px 18px rgba(37,99,235,.4);
  transition:all .3s;
}
#toTop:hover{background:#1d4ed8;transform:translateY(-3px)}

@media (max-width:480px){
  .toc-inner {overflow-x:auto; white-space:nowrap; padding-inline:8px;}
  .toc a {display:inline-block;}
}
</style>
<meta name="description" content="راهنمای تحویل، پرداخت و گارانتی فروشگاه سلام بابا - اطلاعات جامع برای خریداران">
</head>
<body>

<button class="theme-toggle" id="themeToggle" aria-label="تغییر حالت تیره/روشن">🌓</button>

<header class="header" id="siteHeader">
  <div class="logo-stripes"><span class="b"></span><span class="g"></span><span class="y"></span><span class="r"></span></div>
  <h1><?php echo htmlspecialchars($site_title, ENT_QUOTES, 'UTF-8'); ?></h1>
</header>

<nav class="toc" id="topNav">
  <div class="toc-inner">
    <?php foreach ($sections as $sec): ?>
      <a href="#<?php echo htmlspecialchars($sec['id'], ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($sec['title'], ENT_QUOTES, 'UTF-8'); ?></a>
    <?php endforeach; ?>
  </div>
</nav>

<div class="wrap">
  <?php foreach ($sections as $sec): ?>
    <section id="<?php echo htmlspecialchars($sec['id'], ENT_QUOTES, 'UTF-8'); ?>" class="card">
      <details <?php echo !empty($sec['open']) ? 'open' : ''; ?>>
        <summary><?php echo htmlspecialchars($sec['title'], ENT_QUOTES, 'UTF-8'); ?></summary>
        <div class="content"><?php echo $sec['content_html']; ?></div>
      </details>
    </section>
  <?php endforeach; ?>
</div>

<button id="toTop" aria-label="بازگشت به بالا">↑</button>

<script>
// theme toggle (persist)
(function(){
  const root = document.documentElement;
  const saved = localStorage.getItem('theme');
  if(saved){ root.setAttribute('data-theme', saved); }
  document.getElementById('themeToggle').addEventListener('click', function(){
    const cur = root.getAttribute('data-theme') === 'dark' ? 'light' : 'dark';
    if(cur === 'light'){ root.removeAttribute('data-theme'); localStorage.removeItem('theme'); }
    else { root.setAttribute('data-theme', 'dark'); localStorage.setItem('theme','dark'); }
  });
})();

// back-to-top
const btn=document.getElementById('toTop');
window.addEventListener('scroll',()=>btn.style.display=(scrollY>300)?'flex':'none');
btn.addEventListener('click',()=>scrollTo({top:0,behavior:'smooth'}));

// smart sticky nav using IntersectionObserver
const nav=document.getElementById('topNav');
const header=document.getElementById('siteHeader');
const io=new IntersectionObserver((entries)=>{
  entries.forEach(e=>{ nav.classList.toggle('shrink', !e.isIntersecting); });
},{threshold:0});
io.observe(header);

// active link highlight
const sections=document.querySelectorAll('section');
const links=document.querySelectorAll('.toc a');
window.addEventListener('scroll',()=>{
  let current='';
  sections.forEach(sec=>{
    const top=window.scrollY;
    const offset=sec.offsetTop-120;
    const height=sec.offsetHeight;
    if(top>=offset && top<offset+height) current=sec.getAttribute('id');
  });
  links.forEach(a=>{
    a.classList.remove('active');
    if(a.getAttribute('href')==='#'+current) a.classList.add('active');
  });
});
</script>

</body>
</html>
